﻿using System;
using System.Collections.Generic;
using System.Text;
using System.ComponentModel;

namespace IMMeDotNet {

	/// <summary>
	/// Represents a string representation for a field in an <see cref="Enum"/>.
	/// </summary>
	[AttributeUsage(AttributeTargets.Field)]
	public class StringRepresentationAttribute : Attribute {
		/// <summary>
		/// Gets the string representation of the <see cref="Enum"/> field.
		/// </summary>
		public string String { get; private set; }

		/// <summary>
		/// Creates an instance of the <see cref="StringRepresentationAttribute"/> class.
		/// </summary>
		/// <param name="value">The string representation for the <see cref="Enum"/> field.</param>
		public StringRepresentationAttribute(string value) {
			this.String = value;
		}

		/// <summary>
		/// Gets the string representation for an <see cref="Enum"/> field.
		/// </summary>
		/// <param name="value">The <see cref="Enum"/> field to retrieve the string representation for.</param>
		/// <returns>The string representation, or null if one could not be found.</returns>
		public static string GetString(Enum value) {
			string result = null;
			var field = value.GetType().GetField(value.ToString());
			if (field != null) {
				var fieldAttributes = field.GetCustomAttributes(typeof(StringRepresentationAttribute), false);
				if (fieldAttributes.Length == 1) {
					result = ((StringRepresentationAttribute)fieldAttributes[0]).String;
				}
			}
			return result;
		}
	}

	/// <summary>
	/// Represents a character in a string sent to or from the IM-me device.
	/// </summary>
	public enum IMMeChar : byte {
		[StringRepresentation("↕")]
		UpDownArrow = 0x10,
		[StringRepresentation("↓")]
		DownArrow = 0x11,
		[StringRepresentation("↑")]
		UpArrow = 0x12,
		[StringRepresentation("←")]
		LeftArrow = 0x13,
		[StringRepresentation("→")]
		RightArrow = 0x14,
		[StringRepresentation("◦")]
		WhiteCircle = 0x15,
		[StringRepresentation("(A)")]
		SmallA = 0x16,
		[StringRepresentation("•")]
		BlackCircle = 0x19,
		[StringRepresentation("[A]")]
		SmallAInverted = 0x1A,
		[StringRepresentation("©")]
		CopyrightSign = 0x1C,
		[StringRepresentation(" ")]
		Space = 0x20,
		[StringRepresentation("!")]
		ExclamationMark = 0x21,
		[StringRepresentation("\"")]
		QuotationMark = 0x22,
		[StringRepresentation("#")]
		NumberSign = 0x23,
		[StringRepresentation("$")]
		DollarSign = 0x24,
		[StringRepresentation("&")]
		Ampersand = 0x25,
		[StringRepresentation("'")]
		Apostrophe = 0x27,
		[StringRepresentation("(")]
		LeftParenthesis = 0x28,
		[StringRepresentation(")")]
		RightParenthesis = 0x29,
		[StringRepresentation("+")]
		PlusSign = 0x2B,
		[StringRepresentation(",")]
		Comma = 0x2C,
		[StringRepresentation("-")]
		HyphenMinus = 0x2D,
		[StringRepresentation(".")]
		FullStop = 0x2E,
		[StringRepresentation("0")]
		DigitZero = 0x30,
		[StringRepresentation("1")]
		DigitOne = 0x31,
		[StringRepresentation("2")]
		DigitTwo = 0x32,
		[StringRepresentation("3")]
		DigitThree = 0x33,
		[StringRepresentation("4")]
		DigitFour = 0x34,
		[StringRepresentation("5")]
		DigitFive = 0x35,
		[StringRepresentation("6")]
		DigitSix = 0x36,
		[StringRepresentation("7")]
		DigitSeven = 0x37,
		[StringRepresentation("8")]
		DigitEight = 0x38,
		[StringRepresentation("9")]
		DigitNine = 0x39,
		[StringRepresentation(":")]
		Colon = 0x3A,
		[StringRepresentation(";")]
		Semicolon = 0x3B,
		[StringRepresentation("=")]
		EqualsSign = 0x3D,
		[StringRepresentation("?")]
		QuestionMark = 0x3F,
		[StringRepresentation("@")]
		CommercialAt = 0x40,
		[StringRepresentation("A")]
		LatinCapitalLetterA = 0x41,
		[StringRepresentation("B")]
		LatinCapitalLetterB = 0x42,
		[StringRepresentation("C")]
		LatinCapitalLetterC = 0x43,
		[StringRepresentation("D")]
		LatinCapitalLetterD = 0x44,
		[StringRepresentation("E")]
		LatinCapitalLetterE = 0x45,
		[StringRepresentation("F")]
		LatinCapitalLetterF = 0x46,
		[StringRepresentation("G")]
		LatinCapitalLetterG = 0x47,
		[StringRepresentation("H")]
		LatinCapitalLetterH = 0x48,
		[StringRepresentation("I")]
		LatinCapitalLetterI = 0x49,
		[StringRepresentation("J")]
		LatinCapitalLetterJ = 0x4A,
		[StringRepresentation("K")]
		LatinCapitalLetterK = 0x4B,
		[StringRepresentation("L")]
		LatinCapitalLetterL = 0x4C,
		[StringRepresentation("M")]
		LatinCapitalLetterM = 0x4D,
		[StringRepresentation("N")]
		LatinCapitalLetterN = 0x4E,
		[StringRepresentation("O")]
		LatinCapitalLetterO = 0x4F,
		[StringRepresentation("P")]
		LatinCapitalLetterP = 0x50,
		[StringRepresentation("Q")]
		LatinCapitalLetterQ = 0x51,
		[StringRepresentation("R")]
		LatinCapitalLetterR = 0x52,
		[StringRepresentation("S")]
		LatinCapitalLetterS = 0x53,
		[StringRepresentation("T")]
		LatinCapitalLetterT = 0x54,
		[StringRepresentation("U")]
		LatinCapitalLetterU = 0x55,
		[StringRepresentation("V")]
		LatinCapitalLetterV = 0x56,
		[StringRepresentation("W")]
		LatinCapitalLetterW = 0x57,
		[StringRepresentation("X")]
		LatinCapitalLetterX = 0x58,
		[StringRepresentation("Y")]
		LatinCapitalLetterY = 0x59,
		[StringRepresentation("Z")]
		LatinCapitalLetterZ = 0x5A,
		[StringRepresentation("\\")]
		ReverseSolidus = 0x5C,
		[StringRepresentation("_")]
		LowLine = 0x5F,
		[StringRepresentation("a")]
		LatinSmallLetterA = 0x61,
		[StringRepresentation("b")]
		LatinSmallLetterB = 0x62,
		[StringRepresentation("c")]
		LatinSmallLetterC = 0x63,
		[StringRepresentation("d")]
		LatinSmallLetterD = 0x64,
		[StringRepresentation("e")]
		LatinSmallLetterE = 0x65,
		[StringRepresentation("f")]
		LatinSmallLetterF = 0x66,
		[StringRepresentation("g")]
		LatinSmallLetterG = 0x67,
		[StringRepresentation("h")]
		LatinSmallLetterH = 0x68,
		[StringRepresentation("i")]
		LatinSmallLetterI = 0x69,
		[StringRepresentation("j")]
		LatinSmallLetterJ = 0x6A,
		[StringRepresentation("k")]
		LatinSmallLetterK = 0x6B,
		[StringRepresentation("l")]
		LatinSmallLetterL = 0x6C,
		[StringRepresentation("m")]
		LatinSmallLetterM = 0x6D,
		[StringRepresentation("n")]
		LatinSmallLetterN = 0x6E,
		[StringRepresentation("o")]
		LatinSmallLetterO = 0x6F,
		[StringRepresentation("p")]
		LatinSmallLetterP = 0x70,
		[StringRepresentation("q")]
		LatinSmallLetterQ = 0x71,
		[StringRepresentation("r")]
		LatinSmallLetterR = 0x72,
		[StringRepresentation("s")]
		LatinSmallLetterS = 0x73,
		[StringRepresentation("t")]
		LatinSmallLetterT = 0x74,
		[StringRepresentation("u")]
		LatinSmallLetterU = 0x75,
		[StringRepresentation("v")]
		LatinSmallLetterV = 0x76,
		[StringRepresentation("w")]
		LatinSmallLetterW = 0x77,
		[StringRepresentation("x")]
		LatinSmallLetterX = 0x78,
		[StringRepresentation("y")]
		LatinSmallLetterY = 0x79,
		[StringRepresentation("z")]
		LatinSmallLetterZ = 0x7A,
		[StringRepresentation(":)")]
		EmoticonSmiling = 0x80,
		[StringRepresentation(";)")]
		EmoticonWinking = 0x81,
		[StringRepresentation(":(")]
		EmoticonFrowning = 0x82,
		[StringRepresentation(":p")]
		EmoticonTongueStickingOut = 0x83,
		[StringRepresentation(":o")]
		EmoticonSurprised = 0x84,
		[StringRepresentation(":D")]
		EmoticonLaughing = 0x85,
		[StringRepresentation(":!")]
		EmoticonSpeechless = 0x86,
		[StringRepresentation(":'(")]
		EmoticonCrying = 0x87,
		[StringRepresentation(":X")]
		EmoticonSealedLips = 0x88,
		[StringRepresentation("8)")]
		EmoticonGlasses = 0x89,
		[StringRepresentation(":$")]
		EmoticonBlushing = 0x8A,
		[StringRepresentation(":\\")]
		EmoticonSceptical = 0x8B,
		[StringRepresentation(":#")]
		EmoticonRollingEyes = 0x8C,
		[StringRepresentation(":+")]
		EmoticonKissing = 0x8D,
	}
}
